﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Client;
using System;
using System.Linq;
using Microsoft.Crm.Sdk.Messages;
using VA.TMP.DataModel;
using VA.TMP.Integration.VIMT.HealthShare.StateObject;
using VA.TMP.Integration.VIMT.Messages.HealthShare;
using VA.TMP.OptionSets;
using VRM.Integration.Servicebus.Core;

namespace VA.TMP.Integration.VIMT.HealthShare.PipelineSteps.UpdateClinic
{
    /// <summary>
    /// Create and Save CRM entities.
    /// </summary>
    public class CreateAndSaveEntitiesStep : FilterBase<UpdateClinicStateObject>
    {
        /// <summary>
        /// Execute the step.
        /// </summary>
        /// <param name="state">State object.</param>
        public override void Execute(UpdateClinicStateObject state)
        {
            mcs_integrationresult interationResult = null;
            OrganizationServiceContext orgContext = null;
            var debugMessage = string.Empty;
            var integrationStatus = (int)mcs_integrationresultmcs_status.Error;

            try
            {
                if (state.Clinic == null)
                {
                    Logger.Instance.Debug("Clinic in the state object is null. Skipping Clinic Update in VA.TMP.Integration.VIMT.HealthShare.PipelineSteps.UpdateClinic.CreateAndSaveEntitiesStep.Execute");
                }
                else
                {
                    var clinicResource = state.Clinic;
                    orgContext = new OrganizationServiceContext(state.OrganizationServiceProxy);
                    interationResult = CreateIntegrationResult(clinicResource.mcs_resourceId, state, orgContext);

                    if (clinicResource.mcs_RelatedSiteId == null)
                    {
                        debugMessage += $"Error: The site with station number {clinicResource.cvt_StationNumber} does not exists in TMP.\n";
                    }

                    //Create a clinic if TMP doesn't have a clinic available with matching IEN
                    if (clinicResource.mcs_resourceId == null)
                    {
                        debugMessage += "Clinic with IEN supplied in the HealthShare message does not match with any clinic record's IEN available in TMP. ";
                        if (clinicResource.mcs_RelatedSiteId == null)
                        {
                            debugMessage += "Skipping the clinic record creation as site with station number doesnot exist";
                            integrationStatus = (int)mcs_integrationresultmcs_status.Error;
                            throw new ArgumentException(debugMessage, "StationNumber");
                        }
                        else
                        {
                            debugMessage += "Hence creating a new Clinic in TMP";
                            Logger.Instance.Debug(debugMessage);
                            orgContext.AddObject(clinicResource);
                            var response = orgContext.SaveChanges();
                            if (response != null && response.Count > 0)
                            {
                                foreach (var saveChangesResult in response)
                                {
                                    if (saveChangesResult.Response != null)
                                    {
                                        var resourceId = ((Microsoft.Xrm.Sdk.Messages.CreateResponse)saveChangesResult.Response).id;
                                        interationResult.cvt_resourceid = new EntityReference(mcs_resource.EntityLogicalName, resourceId);
                                    }
                                }
                            }
                            integrationStatus = (int)mcs_integrationresultmcs_status.Complete;
                        }
                    }
                    else
                    {
                        debugMessage += $"Clinic with Id: {clinicResource.mcs_resourceId.Value} and IEN: {clinicResource.cvt_ien} available in TMP. " + "Hence creating a new Clinic in TMP";
                        ActivateClinicStatus(clinicResource.mcs_resourceId.Value, state.OrganizationServiceProxy);
                        orgContext.Attach(clinicResource);
                        orgContext.UpdateObject(clinicResource);
                        orgContext.SaveChanges();
                        integrationStatus = (int)mcs_integrationresultmcs_status.Complete;
                    }

                    if (!string.IsNullOrWhiteSpace(state.RequestMessage.ClinicStatus) &&
                        state.RequestMessage.ClinicStatus.ToLowerInvariant() == "i")
                    {
                        UpdateClinicStatus(interationResult.cvt_resourceid.Id, state.OrganizationServiceProxy, false);
                    }
                }
            }
            catch (Exception ex)
            {
                debugMessage += $"HealthShare Update Clinic CreateAndSaveEntitiesStep Pipeline Error:{ex.Message} {ex.InnerException?.Message}";
                throw new Exception(debugMessage, ex.InnerException);
            }
            finally
            {
                UpdateIntegrationResult(interationResult, orgContext, integrationStatus, debugMessage);
            }
        }

        /// <summary>
        /// Activate the clinic Status in case if it's Inactive for updation
        /// </summary>
        /// <param name="clinicId">The Clinic Guid</param>
        /// <param name="svc">The Organization Service object</param>
        private static void ActivateClinicStatus(Guid clinicId, IOrganizationService svc)
        {
            using (var context = new Xrm(svc))
            {
                var clinic = context.mcs_resourceSet.FirstOrDefault(s => s.Id == clinicId);
                if (clinic != null && clinic.statecode == mcs_resourceState.Inactive)
                    UpdateClinicStatus(clinicId, svc, true);
            }
        }

        /// <summary>
        /// Update the clinic Status based on the request
        /// </summary>
        /// <param name="clinicId">The Clinic Guid</param>
        /// <param name="svc">The Organization Service object</param>
        /// <param name="shouldActivate">Should the clinic record be activated</param>
        private static void UpdateClinicStatus(Guid clinicId, IOrganizationService svc, bool shouldActivate)
        {
            try
            {
                var state = new SetStateRequest
                {
                    State = new OptionSetValue(shouldActivate ? (int)mcs_resourceState.Active : (int)mcs_resourceState.Inactive),
                    Status = new OptionSetValue(shouldActivate? (int)mcs_resource_statuscode.Active : (int)mcs_resource_statuscode.Inactive),
                    EntityMoniker = new EntityReference(mcs_resource.EntityLogicalName, clinicId)
                };

                svc.Execute(state);
            }
            catch (Exception ex)
            {
                var message = $"HealthShare Update Clinic CreateAndSaveEntitiesStep Pipeline Error in Method UpdateClinicStatus: {ex.Message} {ex.InnerException}";
                Logger.Instance.Error(message);
                throw new Exception(message, ex.InnerException);
            }
        }

        /// <summary>
        /// Update the integration status record with the provided status and error message (if any)
        /// </summary>
        /// <param name="interationResult">Integration result record object</param>
        /// <param name="orgContext">Organization Service Context</param>
        /// <param name="status">The Integration Result status value to be set</param>
        /// <param name="errorMessage">Error message details (Optional)</param>
        private static void UpdateIntegrationResult(mcs_integrationresult interationResult, OrganizationServiceContext orgContext, int status, string errorMessage = null)
        {
            try
            {
                if (interationResult != null && orgContext != null)
                {
                    interationResult.mcs_error = errorMessage;
                    interationResult.mcs_status = new OptionSetValue(status);
                    orgContext.Attach(interationResult);
                    orgContext.UpdateObject(interationResult);
                    orgContext.SaveChanges();
                }
            }
            catch (Exception ex)
            {
                var message = $"HealthShare Update Clinic CreateAndSaveEntitiesStep Pipeline Error in Method UpdateIntegrationResult: {ex.Message} {ex.InnerException}";
                Logger.Instance.Error(message);
                throw new Exception(message, ex.InnerException);
            }
        }

        /// <summary>
        /// Create the Integration Result Record
        /// </summary>
        /// <param name="resourceId">Associated Resource (Clinic) Record Id</param>
        /// <param name="state">The UpdateClinicStateObject</param>
        /// <param name="orgContext">Organization Context.</param>
        /// <returns>the Integration Result Object created </returns>
        private static mcs_integrationresult CreateIntegrationResult(Guid? resourceId, UpdateClinicStateObject state, OrganizationServiceContext orgContext)
        {
            var newIntegrationResult = new mcs_integrationresult();
            try
            {
                newIntegrationResult.mcs_name = "Update Clinic from HealthShare";
                newIntegrationResult.mcs_integrationrequest = state.SerializedRequestMessage;
                newIntegrationResult.mcs_VimtRequestMessageType = typeof(TmpHealthShareUpdateClinicRequestMessage).FullName;
                newIntegrationResult.mcs_VimtResponseMessageType = typeof(TmpHealthShareUpdateClinicResponseMessage).FullName;
                newIntegrationResult.mcs_VimtMessageRegistryName = MessageRegistry.TmpHealthShareUpdateClinicRequestMessage;

                if (resourceId != null && resourceId != Guid.Empty)
                    newIntegrationResult.cvt_resourceid = new EntityReference(mcs_resource.EntityLogicalName, resourceId.Value);

                orgContext.AddObject(newIntegrationResult);
                orgContext.SaveChanges();
            }
            catch(Exception ex)
            {
                var message = $"HealthShare Update Clinic CreateAndSaveEntitiesStep Pipeline Error in Method CreateIntegrationResult: {ex.Message} {ex.InnerException}";
                Logger.Instance.Error(message);
                throw new Exception(message, ex.InnerException);
            }
            return newIntegrationResult;
        }
    }
}